<?php
/**
 * Master auto-updater for multiple markets (dpboss panel-chart)
 * - Inserts only new results (accepts digits or '*' placeholders)
 * - Skips duplicates (per result_date)
 * - Runs for a list of markets + URLs
 */

date_default_timezone_set("Asia/Kolkata");
error_reporting(E_ALL & ~E_NOTICE);

// --- DB credentials (use your values) ---
$host = "localhost:3306";
$user = "matkabazarfast_matkauser";
$pass = "Ashok@2025";
$db   = "matkabazarfast_matkadb";

$mysqli = new mysqli($host, $user, $pass, $db);
if ($mysqli->connect_error) {
    die("❌ Database connection failed: " . $mysqli->connect_error . PHP_EOL);
}

// --- Markets list: "DISPLAY NAME" => "URL" ---
$markets = [
    //"MILAN MORNING"   => "https://dpboss.boston/panel-chart-record/milan-morning.php",
    //"KALYAN MORNING"  => "https://dpboss.boston/panel-chart-record/kalyan-morning.php",
    //"SRIDEVI"         => "https://dpboss.boston/panel-chart-record/sridevi.php",
    //"TIME BAZAR"      => "https://dpboss.boston/panel-chart-record/time-bazar.php",
    //"MADHUR DAY"      => "https://dpboss.boston/panel-chart-record/madhur-day.php",
    //"BOMBAY DAY"      => "https://dpboss.boston/panel-chart-record/bombay-day.php",
    //"MILAN DAY"       => "https://dpboss.boston/panel-chart-record/milan-day.php",
   // "RAJDHANI DAY"    => "https://dpboss.boston/panel-chart-record/rajdhani-day.php",
    //"SUPREME DAY"     => "https://dpboss.boston/panel-chart-record/supreme-day.php",
    //"KALYAN"          => "https://dpboss.boston/panel-chart-record/kalyan.php",
    //"SRIDEVI NIGHT"   => "https://dpboss.boston/panel-chart-record/sridevi-night.php",
    //"MADHUR NIGHT"    => "https://dpboss.boston/panel-chart-record/madhur-night.php",
    //"SUPREME NIGHT"   => "https://dpboss.boston/panel-chart-record/supreme-night.php",
    //"MILAN NIGHT"     => "https://dpboss.boston/panel-chart-record/milan-night.php",
    //"KALYAN NIGHT"    => "https://dpboss.boston/panel-chart-record/kalyan-night.php",
    //"RAJDHANI NIGHT"  => "https://dpboss.boston/panel-chart-record/rajdhani-night.php",
    //"MAIN BAZAR"      => "https://dpboss.boston/panel-chart-record/main-bazar.php",
    //"TIME BAZAR DAY"    => "https://dpboss.boston/panel-chart-record/time-bazar-day.php",
    "NIGHT TIME BAZAR"    => "https://dpboss.boston/panel-chart-record/night-time-bazar.php"
    ];

// Helper: sanitize table name from market display name
function market_to_table($marketName) {
    // Replace spaces and non-alphanumeric with underscore, uppercase
    $t = preg_replace('/[^\w]/', '_', $marketName);
    $t = strtoupper($t);
    return "game_" . $t;
}

// Accept digits or star placeholders. Valid day lengths: open=3, jodi=2, close=3
function normalize_cell($raw) {
    // remove whitespace and keep digits and stars only
    $val = preg_replace('/[^\d\*]/', '', $raw);
    return $val;
}

$global_total = 0;

foreach ($markets as $marketName => $url) {
    echo "---- Processing: $marketName  $url" . PHP_EOL;
    $tableName = market_to_table($marketName);

    // Safety: ensure tableName contains only allowed chars
    if (!preg_match('/^game_[A-Z0-9_]+$/', $tableName)) {
        echo "❌ Skipping invalid table name: $tableName" . PHP_EOL;
        continue;
    }

    // fetch page
    $html = @file_get_contents($url);
    if (!$html) {
        echo "❌ Failed to load URL: $url" . PHP_EOL;
        continue;
    }

    libxml_use_internal_errors(true);
    $dom = new DOMDocument();
    $dom->loadHTML($html);
    libxml_clear_errors();
    $xpath = new DOMXPath($dom);
    $rows = $xpath->query("//table//tbody//tr");

    $inserted = 0;
    $marketVar = $marketName; // param for insert

    foreach ($rows as $row) {
        $cells = $row->getElementsByTagName("td");
        if ($cells->length < 2) continue;

        // date cell text may contain <br/> so normalize whitespace
        $range = preg_replace('/\s+/', '', $cells->item(0)->textContent);
        if (!preg_match("/(\d{2}\/\d{2}\/\d{4})to(\d{2}\/\d{2}\/\d{4})/", $range, $m)) {
            continue;
        }

        $start = DateTime::createFromFormat('d/m/Y', $m[1]);
        $end   = DateTime::createFromFormat('d/m/Y', $m[2]);

        // build dates array for the week
        $dates = [];
        while ($start <= $end) {
            $dates[] = $start->format('Y-m-d');
            $start->modify('+1 day');
        }

        // iterate columns: every 3 td => open,jodi,close
        $dayIndex = 0;
        for ($i = 1; $i + 2 < $cells->length && $dayIndex < 7; $i += 3) {
            $rawOpen  = trim($cells->item($i)->textContent);
            $rawJodi  = trim($cells->item($i + 1)->textContent);
            $rawClose = trim($cells->item($i + 2)->textContent);
            $date     = $dates[$dayIndex] ?? null;
            if (!$date) { $dayIndex++; continue; }

            $open  = normalize_cell($rawOpen);
            $jodi  = normalize_cell($rawJodi);
            $close = normalize_cell($rawClose);

            // Accept either digits or stars in the exact required length
            $okOpen  = preg_match('/^(?:\d{3}|\*{3})$/', $open);
            $okJodi  = preg_match('/^(?:\d{2}|\*{2})$/', $jodi);
            $okClose = preg_match('/^(?:\d{3}|\*{3})$/', $close);

            if ($okOpen && $okJodi && $okClose) {
                $formatted = $open . '-' . $jodi . '-' . $close;

                // duplicate check by result_date
                // prepare dynamic SQL safely by escaping table name (we validated above)
                $check_sql = "SELECT COUNT(*) FROM `$tableName` WHERE result_date = ?";
                $check = $mysqli->prepare($check_sql);
                if (!$check) {
                    echo "❌ Prepare failed for table $tableName: " . $mysqli->error . PHP_EOL;
                    continue;
                }
                $check->bind_param("s", $date);
                $check->execute();
                $check->bind_result($count);
                $check->fetch();
                $check->close();

                if ($count == 0) {
                    $ins_sql = "INSERT INTO `$tableName` (name, result, result_date, date, status) VALUES (?, ?, ?, NOW(), '0')";
                    $stmt = $mysqli->prepare($ins_sql);
                    if (!$stmt) {
                        echo "❌ Insert prepare failed for $tableName: " . $mysqli->error . PHP_EOL;
                        continue;
                    }
                    $stmt->bind_param("sss", $marketVar, $formatted, $date);
                    if ($stmt->execute()) {
                        $inserted++;
                        $global_total++;
                    } else {
                        echo "❌ Insert failed for $tableName ($date): " . $stmt->error . PHP_EOL;
                    }
                    $stmt->close();
                }
            }
            // else skip incomplete/malformed cells (we don't insert partials)
            $dayIndex++;
        }
    } // foreach rows

    echo "✅ $marketName: New records inserted: $inserted" . PHP_EOL;
} // foreach markets

$mysqli->close();

echo PHP_EOL . "==== All markets completed. Total new rows inserted: $global_total ====" . PHP_EOL;
?>
